import { CONFIG, apiUrl } from "../config.js";
import { buildEmail, normalizePayloadData, vipText, postForm } from "../lib/auth_util.js";

const REGISTER_URL = apiUrl(CONFIG.API.REGISTER);
const SEND_CODE_URL = apiUrl(CONFIG.API.SENDCODE);

function $(id) { return document.getElementById(id); }

function setStatus(text) {
  const el = $("status");
  if (el) el.textContent = text;
}

// 发送按钮倒计时（保持你之前的逻辑）
let sendTimer = null;
let cooldown = 0;

function setSendBtn(enabled, text) {
  const btn = $("email_send");
  if (!btn) return;
  btn.disabled = !enabled;
  btn.textContent = text;
}

function startCooldown(seconds = 60) {
  cooldown = seconds;
  setSendBtn(false, `已发送(${cooldown}s)`);
  clearInterval(sendTimer);
  sendTimer = setInterval(() => {
    cooldown -= 1;
    if (cooldown <= 0) {
      clearInterval(sendTimer);
      sendTimer = null;
      setSendBtn(true, "发送");
    } else {
      setSendBtn(false, `已发送(${cooldown}s)`);
    }
  }, 1000);
}

async function sendEmailCode() {
  const usernameInput = $("username")?.value?.trim() || "";
  const ext = $("login_mail_ext")?.value || "";
  const email = buildEmail(usernameInput, ext);

  if (!email) {
    setStatus("请输入邮箱用户名");
    return;
  }

  setStatus("验证码发送中...");
  setSendBtn(false, "发送中...");

  try {
    const r = await postForm(SEND_CODE_URL, { email });
    const payload = r.json;

    if (!payload) {
      setStatus("验证码接口返回不是 JSON：\n" + r.raw);
      setSendBtn(true, "发送");
      return;
    }

    // 已注册：{"ret":0,"msg":"此邮箱已经注册"}
    if (payload.ret === 0 || payload.ret === "0") {
      setStatus(payload.msg || "此邮箱已经注册");
      setSendBtn(true, "发送");
      return;
    }

    setStatus(payload.msg || "邮件验证码已发送，请查收邮箱（可能在垃圾箱）");
    startCooldown(60);
  } catch (e) {
    setStatus("发送失败：" + (e?.message || String(e)));
    setSendBtn(true, "发送");
  }
}

async function doRegister() {
  const usernameInput = $("username")?.value?.trim() || "";
  const ext = $("login_mail_ext")?.value || "";
  const email = buildEmail(usernameInput, ext);

  const code = $("email_code")?.value?.trim() || "";
  const p1 = $("password")?.value || "";
  const p2 = $("password2")?.value || "";

  if (!email || !p1 || !p2) return setStatus("请完整填写注册信息");
  if (!code) return setStatus("请输入邮件验证码");
  if (p1 !== p2) return setStatus("两次密码不一致");

  setStatus("注册中...");

  try {
    // ⚠️ email_code 字段名按你后端为准
    const r = await postForm(REGISTER_URL, {
      name: usernameInput,
      email,
      password: p1,
      confirm_password: p1,
      emailcode: code,
      invite_code: '',
      tos: true,
      from: 'browser'
    });

    const payload = r.json;
    if (!payload) return setStatus("注册接口返回不是 JSON：\n" + r.raw);

    // 你注册成功目前仍是 success:"1" / 或也可能改成 ret:"1"
    // 这里兼容两种：
    const ok = (payload.success === "1") || (payload.ret === "1");
    if (!ok) {
      return setStatus(payload.message || payload.msg || "注册失败");
    }

    const data = normalizePayloadData(payload);
    if (!data) return setStatus("注册成功但 data 解码失败");

    const username = data.username || email;
    const email = data.email;
    const token = data.token;
    const expire = data.expire;
    const vip = data.vip ?? "0";
    const speed = data.speeed ?? "N/A";
    const quota = data.quota ?? "N/A";

    if (!token) return setStatus("注册成功但未返回 token");

    await chrome.storage.local.set({ loggedIn: true, email, username, token, expire, vip, speed, quota });
    
    chrome.runtime.sendMessage({ type: "AUTH_UPDATED" });
    chrome.runtime.sendMessage({ type: "REFRESH" });

    setStatus(`注册成功并已登录：${username}（${vipText(vip)}）`);
    location.href = chrome.runtime.getURL("pages/welcome.html");
    // window.close();
  } catch (e) {
    setStatus("请求失败：" + (e?.message || String(e)));
  }
}

document.addEventListener("DOMContentLoaded", () => {
  $("back")?.addEventListener("click", () => window.close());
  $("email_send")?.addEventListener("click", sendEmailCode);
  $("register")?.addEventListener("click", doRegister);
});
