import { CONFIG, apiUrl } from "./config.js";

let PRODUCTS = [];
let PAYMENTS = []; // 扁平化：[{payId, typeKey, label}]
let SELECTED_PRODUCT_ID = null;
let SELECTED_QTY = 1;
let SELECTED_PAY_VALUE = null; // "payId|typeKey"

// ====== 你原来的工具函数（保留） ======
function pad2(n) {
  n = Number(n);
  return n < 10 ? "0" + n : "" + n;
}
function getPromoTotalPrice(product, qty) {
  // 返回：Number 或 NaN
  if (!product) return NaN;
  const q = Number(qty || 1);

  // 单价
  const unit = Number(product.price);
  const unitPrice = Number.isFinite(unit) ? unit : Number(product.price || 0);

  // qty=1 直接返回单价
  if (q <= 1) return unitPrice;

  // promotion 可能是 [] 或 {}，你接口有两种
  const promos = Array.isArray(product.promotion) ? product.promotion : [];

  // promotion.price 是“合计价”
  const hit = promos.find(x => Number(x.quantity) === q && Number.isFinite(Number(x.price)));
  if (hit) return Number(hit.price);

  // 没命中促销 → 兜底按单价*数量
  return Number.isFinite(unitPrice) ? unitPrice * q : NaN;
}

function getPromoLabel(product, qty) {
  if (!product) return "";
  const q = Number(qty || 1);
  if (q <= 1) return "";

  const promos = Array.isArray(product.promotion) ? product.promotion : [];
  const hit = promos.find(x => Number(x.quantity) === q && Number.isFinite(Number(x.price)));
  if (!hit) return "";

  const unit = Number(product.price);
  const unitPrice = Number.isFinite(unit) ? unit : Number(product.price || 0);
  const saved = Number.isFinite(unitPrice) ? (unitPrice * q - Number(hit.price)) : NaN;
  if (!Number.isFinite(saved) || saved <= 0) return "已使用促销价";

  return `（省￥${saved.toFixed(2)}）`;
}

function formatExpire(expire) {
  if (!expire) return "未设置";
  let t = Number(expire);
  if (!Number.isFinite(t) || t <= 0) return "无效";
  if (t > 1e12) t = Math.floor(t / 1000);
  const d = new Date(t * 1000);
  return `${d.getFullYear()}-${pad2(d.getMonth()+1)}-${pad2(d.getDate())} ${pad2(d.getHours())}:${pad2(d.getMinutes())}`;
}

function vipText(vip) {
  if (vip === undefined || vip === null || vip === 0) return "免费用户";
  const n = Number(vip);
  if (Number.isFinite(n)) return `VIP${n}`;
  return String(vip);
}

function quotaText(m) {
  if (m === undefined || m === null || m === "") return "-";
  const n = Number(m);
  if (!Number.isFinite(n)) return String(m);
  return `${n.toFixed(2)}`;
}

// ====== storage 账号信息 ======
async function loadAccount(st) {


  if (!st.loggedIn) {
    location.href = chrome.runtime.getURL("pages/user/login.html");
    return;
  }

  document.getElementById("u_name").textContent = st.username || "-";
  document.getElementById("u_vip").textContent = vipText(st.vip);
  document.getElementById("u_expire").textContent = formatExpire(st.expire);
  document.getElementById("u_quota").textContent = quotaText(st.quota);
}

// ====== 拉取 product_list（ret=1 才成功） ======
async function fetchProductList() {
  const url = apiUrl(CONFIG.API.PRODUCTLIST);
  const resp = await fetch(url, { method: "GET", headers: { "Accept": "application/json" } });
  if (!resp.ok) throw new Error(`HTTP ${resp.status}`);

  const json = await resp.json();
  if (String(json?.ret) !== "1" && Number(json?.ret) !== 1) {
    throw new Error(json?.msg || "获取数据失败");
  }

  const data = json?.data || {};
  const product = Array.isArray(data.product) ? data.product : [];
  const payment = data.payment || {};

  return { product, payment };
}

// ====== 产品过滤 & 排序 ======
function normalizeProducts(list) {
  // 过滤 status=1 且 content.class != 0
  const filtered = list.filter(p => {
    if (Number(p.status) !== 1) return false;
    const cls = Number(p?.content?.class ?? 0);
    return cls !== 0;
  });

  // 排序：position 小的在前，其次 id
  filtered.sort((a, b) => {
    const pa = Number(a.position ?? 9999);
    const pb = Number(b.position ?? 9999);
    if (pa !== pb) return pa - pb;
    return Number(a.id) - Number(b.id);
  });

  return filtered;
}

// ====== 支付方式扁平化 ======
function normalizePayments(paymentObj) {
  // payment: { pay1:{id,type:{...}}, pay2:{...} }
  const out = [];
  for (const k of Object.keys(paymentObj || {})) {
    const item = paymentObj[k];
    const payId = item?.id;
    const typeMap = item?.type || {};
    if (!payId) continue;

    for (const typeKey of Object.keys(typeMap)) {
      out.push({
        payId: String(payId),
        typeKey: String(typeKey),
        label: String(typeMap[typeKey] || typeKey)
      });
    }
  }
  return out;
}

// ====== UI 渲染：套餐 ======
function productTitle(p) {
  const price = Number(p.price);
  const priceText = Number.isFinite(price) ? `￥${price.toFixed(2)}` : `￥${p.price}`;
  return `${priceText} · ${p.name || "套餐"}`;
}

function productDesc(p) {
  const c = p.content || {};
  const time = c.time ? `${c.time}天` : "";
  const bw = c.bandwidth ? `${c.bandwidth}GB` : "";
  const cls = (c.class !== undefined && c.class !== null) ? `VIP${c.class}` : "";
  const ip = c.ip_limit ? `IP ${c.ip_limit}` : "";
  const speed = c.speed_limit ? `限速 ${c.speed_limit}` : "";
  return [ [time, bw, cls].filter(Boolean).join(" · "), [ip, speed].filter(Boolean).join(" · ") ]
    .filter(Boolean)
    .join(" / ");
}

function renderPlans() {
  const box = document.getElementById("plans");
  box.innerHTML = "";

  if (!PRODUCTS.length) {
    box.innerHTML = `<div class="empty">暂无可用套餐</div>`;
    return;
  }

  if (!SELECTED_PRODUCT_ID) {
    SELECTED_PRODUCT_ID = String(PRODUCTS[0].id);
  }

  for (const p of PRODUCTS) {
    const id = String(p.id);
    const label = document.createElement("label");
    label.className = "plan";

    label.innerHTML = `
      <input type="radio" name="plan" value="${id}" ${id === SELECTED_PRODUCT_ID ? "checked" : ""} />
      <div class="plan-main">
        <div class="plan-price">${productTitle(p)}</div>
        <div class="plan-desc">${productDesc(p)}</div>
      </div>
      <div class="plan-tag">VIP${p?.content?.class ?? ""}</div>
    `;
    box.appendChild(label);
  }

  box.querySelectorAll('input[name="plan"]').forEach(el => {
    el.addEventListener("change", () => {
      SELECTED_PRODUCT_ID = el.value;
      SELECTED_QTY = 1; // 切换套餐时默认数量回到1
      renderPromotions(); // 重新渲染促销选项
      updateSummary();
    });
  });
}

// ====== UI 渲染：促销（quantity选择） ======
function getSelectedProduct() {
  return PRODUCTS.find(p => String(p.id) === String(SELECTED_PRODUCT_ID)) || null;
}

function renderPromotions() {
  const box = document.getElementById("promoBox");
  if (!box) return;

  const p = getSelectedProduct();
  box.innerHTML = "";

  if (!p) return;

  const promos = Array.isArray(p.promotion) ? p.promotion : [];
  // promotion 可能是 {}，你示例里第二个就是 {}
  const hasPromo = promos.length > 0;

  const wrap = document.createElement("div");
  wrap.className = "promo-wrap";

  // 基础数量（1个）
  const basePrice = Number(p.price);
  const baseText = Number.isFinite(basePrice) ? `￥${basePrice.toFixed(2)}` : `￥${p.price}`;

  let html = `
    <div class="promo-title">购买数量（含促销）</div>
    <div class="promo-options">
      <label class="promo-item">
        <input type="radio" name="qty" value="1" checked />
        <div>1 个月</div>
        <div class="muted">默认为单月套餐 ${baseText} / 月</div>
      </label>
  `;

  if (hasPromo) {
    for (const pr of promos) {
      const q = Number(pr.quantity);
      const price = Number(pr.price);
      if (!Number.isFinite(q) || q <= 1) continue;

      const totalText = Number.isFinite(price) ? `￥${price.toFixed(2)}` : `￥${pr.price}`;
      const per = Number.isFinite(price) ? (price / q) : NaN;
      const perText = Number.isFinite(per) ? `≈ ￥${per.toFixed(2)} / 月` : "";

      html += `
        <label class="promo-item">
          <input type="radio" name="qty" value="${q}" />
          <div>${q} 个月</div>
          <div class="muted">${totalText}（合计） ${perText}</div>
        </label>
      `;
    }
  } else {
    html += `<div class="muted" style="padding:8px 4px;">当前套餐暂无促销</div>`;
  }

  html += `</div>`;
  wrap.innerHTML = html;
  box.appendChild(wrap);

  box.querySelectorAll('input[name="qty"]').forEach(el => {
    el.addEventListener("change", () => {
      SELECTED_QTY = Number(el.value) || 1;
      updateSummary();
    });
  });
}

// ====== UI 渲染：支付方式 ======
function renderPayments() {
  const box = document.getElementById("paymentBox");
  box.innerHTML = "";

  if (!PAYMENTS.length) {
    box.innerHTML = `<div class="empty">暂无可用支付方式</div>`;
    return;
  }

  // 默认选第一个
  if (!SELECTED_PAY_VALUE) {
    SELECTED_PAY_VALUE = `${PAYMENTS[0].payId}|${PAYMENTS[0].typeKey}`;
  }

  let html = `<div class="pay-title">选择支付方式</div>`;
  html += `<div class="pay-options">`;

  for (const it of PAYMENTS) {
    const val = `${it.payId}|${it.typeKey}`;
    html += `
      <label class="payway">
        <input type="radio" name="payway" value="${val}" ${val === SELECTED_PAY_VALUE ? "checked" : ""} />
        <div class="payway-name">${it.label}</div>
        <div class="muted">通道ID：${it.payId}</div>
      </label>
    `;
  }

  html += `</div>`;
  box.innerHTML = html;

  box.querySelectorAll('input[name="payway"]').forEach(el => {
    el.addEventListener("change", () => {
      SELECTED_PAY_VALUE = el.value;
      updateSummary();
    });
  });
}

function getSelectedPayment() {
  const el = document.querySelector('input[name="payway"]:checked');
  const val = el ? el.value : SELECTED_PAY_VALUE;
  if (!val) return null;

  const [payId, typeKey] = val.split("|");
  const found = PAYMENTS.find(x => x.payId === payId && x.typeKey === typeKey);
  return found ? { payId, typeKey, label: found.label } : { payId, typeKey, label: val };
}

// ====== 汇总更新 ======
function updateSummary() {
  const p = getSelectedProduct();
  const pay = getSelectedPayment();

  const sumPlanEl = document.getElementById("sum_plan");
  const sumWayEl = document.getElementById("sum_way");

  if (sumPlanEl) {
    if (!p) {
      sumPlanEl.textContent = "-";
    } else {
      const qty = Number(SELECTED_QTY || 1);

      const total = getPromoTotalPrice(p, qty);
      const totalText = Number.isFinite(total) ? `￥${total.toFixed(2)}` : "-";

      const unit = Number(p.price);
      const unitPrice = Number.isFinite(unit) ? unit : Number(p.price || 0);
      const unitText = Number.isFinite(unitPrice) ? `￥${unitPrice.toFixed(2)}` : `￥${p.price}`;

      const promoLabel = getPromoLabel(p, qty);

      // 你想简洁点就删掉 unitText / promoLabel 的部分
      sumPlanEl.textContent =
        `${p.name} · ${qty}个月 · 单价${unitText} · 合计${totalText}` +
        (promoLabel ? ` · ${promoLabel}` : "");
    }
  }

  if (sumWayEl) {
    sumWayEl.textContent = pay ? (pay.label || "-") : "-";
  }
}


// ====== 下单提交（DOUPGRADE）并跳转 ======
async function submitUpgradeOrder(st) {
  const p = getSelectedProduct();
  const pay = getSelectedPayment();

  if (!p) {
    alert("请选择套餐");
    return;
  }
  if (!pay) {
    alert("请选择支付方式");
    return;
  }

  const url = apiUrl(CONFIG.API.DOUPGRADE);


  const payload = {
    coupon : '',
    type : 'product',
    email : st.email,
    product_id: Number(p.id),
    quantity: Number(SELECTED_QTY || 1),
    payment_id: Number(pay.payId),
    payment_type: String(pay.typeKey)
  };

  const resp = await fetch(url, {
    method: "POST",
    headers: { "Content-Type": "application/json", "Accept": "application/json" },
    body: JSON.stringify(payload)
  });

  if (!resp.ok) throw new Error(`HTTP ${resp.status}`);

  const targetUrl = resp.headers.get('HX-Redirect');
  chrome.tabs.create({ url: targetUrl });
  return;
  //end------------------------------
  const json = await resp.json();
  if (String(json?.ret) !== "1" && Number(json?.ret) !== 1) {
    throw new Error(json?.msg || "下单失败");
  }

  const payUrl = json?.url;
  if (!payUrl) throw new Error("下单成功但未返回支付地址");

  // 跳转到支付地址（新标签更稳）
  //alert(CONFIG.API.BASE + payUrl);
  chrome.tabs.create({ url: CONFIG.API.BASE + payUrl });
}

// ====== 退出/返回/按钮绑定 ======
async function doLogout() {
  await chrome.storage.local.remove(["loggedIn", "email","username", "token", "vip", "expire", "quota", "speed", "money"]);
  chrome.runtime.sendMessage({ type: "AUTH_UPDATED" });
  location.href = chrome.runtime.getURL("pages/user/login.html");
}

function bindUI(st) {
  document.getElementById("btn_back")?.addEventListener("click", () => {
    location.href = chrome.runtime.getURL("pages/welcome.html");
  });

  document.getElementById("btn_to_welcome")?.addEventListener("click", () => {
    location.href = chrome.runtime.getURL("pages/welcome.html");
  });

  document.getElementById("btn_logout")?.addEventListener("click", doLogout);

  document.getElementById("btn_pay_now")?.addEventListener("click", async () => {
    try {
      await submitUpgradeOrder(st);
    } catch (e) {
      alert(String(e.message || e));
    }
  });
}

// ====== 启动 ======
async function init() {
  const st = await chrome.storage.local.get([
  "loggedIn", "email", "username", "vip", "expire", "quota", "speed", "money"
  ]);
  await loadAccount(st);
  bindUI(st);

  // 加载产品+支付方式
  const plansBox = document.getElementById("plans");
  const payBox = document.getElementById("paymentBox");
  if (plansBox) plansBox.innerHTML = `<div class="loading">正在加载套餐…</div>`;
  if (payBox) payBox.innerHTML = `<div class="loading">正在加载支付方式…</div>`;

  try {
    const { product, payment } = await fetchProductList();

    PRODUCTS = normalizeProducts(product);
    PAYMENTS = normalizePayments(payment);

    renderPlans();
    renderPromotions();
    renderPayments();
    updateSummary();
  } catch (e) {
    if (plansBox) plansBox.innerHTML = `<div class="empty">加载失败：${String(e.message || e)}</div>`;
    if (payBox) payBox.innerHTML = `<div class="empty">加载失败：${String(e.message || e)}</div>`;
  }
}

document.addEventListener("DOMContentLoaded", init);
